# pylint: disable=unused-import,abstract-method,unused-argument,no-member

##########################################################################################
#### Imports
##########################################################################################

import logging
import rpdb
import random
import re
import time
import os
import warnings

from math import ceil
from typing import Union, List
from dallinger import db
from flask import Markup
from statistics import mean
from sqlalchemy import exc as sa_exc

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__file__)

import psynet.experiment
from psynet.timeline import (
    Timeline,
    join,
    PageMaker
)
from psynet.page import (
    InfoPage,
    SuccessfulEndPage,
    NumberInputPage,
    NAFCPage,
    TextInputPage
)
from psynet.trial.non_adaptive import (
    StimulusSpec,
    StimulusVersionSpec,
    StimulusSet,
    NonAdaptiveTrialMaker,
    NonAdaptiveTrial
)
from psynet.modular_page import ModularPage, Prompt, NAFCControl
from psynet.ishihara import colour_blind_test
from psynet.colour_vocab import colour_vocab_test

CONFIG = {
    "num_experiment_trials": 60,
    "num_repeat_trials": 8
}

class ColourPrompt(Prompt):
    macro = "colour_prompt"
    external_template = "colour_prompt.html"

    def __init__(self, colour, **kwargs):
        super().__init__(**kwargs)
        self.colour = colour

    @property
    def metadata(self):
        return {
            **super().metadata,
            "colour": self.colour
        }

demographics = join(
    InfoPage(
        "First we need to ask some quick questions about you.",
        time_estimate=5
    ),
    NumberInputPage(
        label='age',
        prompt='What is your age, in years?',
        time_estimate=5
    ),
    NAFCPage(
        label='gender',
        prompt='With what gender do you most identify yourself?',
        time_estimate=5,
        choices=['Male', 'Female', 'Other'],
        arrange_vertically=True
    ),
    NAFCPage(
        label='education',
        prompt='What is your highest educational qualification?',
        time_estimate=7,
        choices=['None', 'Elementary school', 'Middle school', 'High school', 'Bachelor', 'Master', 'PhD'],
        arrange_vertically=True
    ),
    TextInputPage(
        "country",
        "What country are you from?",
        time_estimate=5,
        one_line=True
    ),
    TextInputPage(
        "mother_tongue",
        "What is your mother tongue?",
        time_estimate=5,
        one_line=True
    )
)

final_questionnaire = join(
    TextInputPage(
        "strategy",
        """
        Please tell us in a few words about your experience taking the task.
        What was your strategy?
        Did you find the task easy or difficult?
        Did you find it interesting or boring?
        """,
        time_estimate=20,
        one_line=False
    ),
    TextInputPage(
        "technical",
        """
        Did you experience any technical problems during the task?
        If so, please describe them.
        """,
        time_estimate=10,
        one_line=False
    )
)

def make_timeline():
    return Timeline(
        demographics,
        colour_vocab_test(),
        colour_blind_test(),
        instructions(),
        InfoPage(
            f"""
            You will take up to
            {CONFIG['num_experiment_trials'] + CONFIG['num_repeat_trials']} trials
            where you have to answer this question. Remember to pay careful attention
            in order to get the best bonus!
            """,
            time_estimate=5
        ),
        make_experiment_trials(),
        final_questionnaire,
        SuccessfulEndPage()
    )

def instructions():
    return join(
        InfoPage(Markup(
            """
            <p>
                In each trial of this experiment you will be presented with a word
                and a colour. Your task will be to judge how well the colour
                matches the word.
            </p>
            <p>
                You will have four response options, ranging from 'Not at all' to 'Very much'.
                Choose the one you think is most appropriate.
            </p>
            """),
            time_estimate=5
        ),
        InfoPage(
            """
            The quality of your responses will be automatically monitored,
            and you will receive a bonus at the end of the experiment
            in proportion to your quality score. The best way to achieve
            a high score is to concentrate and give each trial your best attempt.
            """,
            time_estimate=5
        )
    )

def get_stimulus_set(phase: str):
    import pandas as pd
    phase = "experiment"
    df = pd.read_csv("colour-states.csv")
    stimuli = [
        StimulusSpec(
            {
                "old_network_id": record["network_id"],
                "old_node_id": record["node_id"],
                "degree": record["degree"],
                "iteration": record["iteration"],
                "colour": record["colour"],
                "target": record["target"],
                "type": record["type"]
            },
            phase=phase
        )
        for record in df.to_dict("records")
    ]
    return StimulusSet(stimuli)

nafc_choices = [
    (1, "(1) Not at all"),
    (2, "(2) A little"),
    (3, "(3) Quite a lot"),
    (4, "(4) Very much")
]

class RatingControl(NAFCControl):
    def __init__(self):
        super().__init__(
            choices=[x[0] for x in nafc_choices],
            labels=[x[1] for x in nafc_choices],
            min_width="125px"
        )

    def format_answer(self, raw_answer, **kwargs):
        return int(raw_answer)

class CustomTrial(NonAdaptiveTrial):
    __mapper_args__ = {"polymorphic_identity": "custom_trial"}

    def show_trial(self, experiment, participant):
        prompt = Markup(
            f"""
            <p>How well does the colour match the following word:</p>
            <p><strong>{self.definition['target']}</strong></p>
            """
        )

        return ModularPage(
            "custom_trial",
            ColourPrompt(self.definition["colour"], text=prompt, text_align="center"),
            RatingControl()
        )

class CustomTrialMaker(NonAdaptiveTrialMaker):
    give_end_feedback_passed = False
    performance_threshold = -1.0

    def compute_bonus(self, score, passed):
        if self.phase == "practice":
            return 0.0
        elif self.phase == "experiment":
            if score is None:
                return 0.0
            else:
                return max(0.0, score)
        else:
            raise NotImplementedError

def make_experiment_trials():
    return CustomTrialMaker(
        trial_class=CustomTrial,
        phase="experiment",
        stimulus_set=get_stimulus_set(phase="experiment"),
        time_estimate_per_trial=3,
        new_participant_group=True,
        recruit_mode="num_trials",
        target_num_participants=None,
        target_num_trials_per_stimulus=5,
        max_trials_per_block=CONFIG["num_experiment_trials"],
        allow_repeated_stimuli=False,
        max_unique_stimuli_per_block=None,
        active_balancing_within_participants=True,
        active_balancing_across_participants=True,
        check_performance_at_end=True,
        check_performance_every_trial=False,
        fail_trials_on_premature_exit=True,
        fail_trials_on_participant_performance_check=False,
        num_repeat_trials=CONFIG["num_repeat_trials"]
    )

class Exp(psynet.experiment.Experiment):
    timeline = make_timeline()

    def __init__(self, session=None):
        super().__init__(session)
        self.initial_recruitment_size = 30

extra_routes = Exp().extra_routes()
